﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace DocumentProcessing
{
    class DocumentProcessor
    {
        public event EventHandler Processing;
        public event EventHandler Processed;

        public Func<Document, string> LogTextProvider
        {
            get;
            set;
        }


        class ActionCheckPair
        {
            public Action<Document> Action { get; set; }
            public Predicate<Document> QuickCheck { get; set; }
        }

        private readonly List<ActionCheckPair> processes =
            new List<ActionCheckPair>();

        public void AddProcess(Action<Document> action)
        {
            AddProcess(action, null);
        }

        public void AddProcess(Action<Document> action,
            Predicate<Document> quickCheck)
        {
            processes.Add(
            new ActionCheckPair { Action = action, QuickCheck = quickCheck });
        }

        // Listing 5-21. Zgłaszanie zdarzeń
        public void Process(Document doc)
        {
            OnProcessing(EventArgs.Empty);
            // Najpierw przeprowadzamy szybką weryfikację dokumentu.
            foreach (ActionCheckPair process in processes)
            {
                if (process.QuickCheck != null && !process.QuickCheck(doc))
                {
                    Console.WriteLine("Przetwarzanie nie zakończy się pomyślnie.");
                    if (LogTextProvider != null)
                    {
                        Console.WriteLine(LogTextProvider(doc));
                    }
                    OnProcessed(EventArgs.Empty);
                    return;
                }
            }

            // Teraz wykonujemy akcję
            foreach (ActionCheckPair process in processes)
            {
                process.Action(doc);
                if (LogTextProvider != null)
                {
                    Console.WriteLine(LogTextProvider(doc));
                }
            }
            OnProcessed(EventArgs.Empty);
        }

        private void OnProcessing(EventArgs e)
        {
            if (Processing != null)
            {
                Processing(this, e);
            }
        }

        private void OnProcessed(EventArgs e)
        {
            if (Processed != null)
            {
                Processed(this, e);
            }
        }
    }
}